#!/usr/bin/python3
#  coding: utf-8
#
#          remeshAndMapping.py
#
#       dynamicMeshを使っているsolverでremesh、mappingする。
#       latestTimeのpatch形状からstlを作成する。（surfaceMeshTriangulateより)
#       surfaceFeatures(surfaceFeatureExtract)で特徴線再取得、
#       blockMesh, snappyHexMeshを実行。
#       newMeshにmappingして、mappoing結果を戻す
#       このために、予め以下のdictを準備しておく。
#           surfaceFeaturesDict(surfaceFeatureExtractDict), 
#           blockMeshDict, snappyHexMeshDict, mapFieldsDict
#
#   25/07/25    新規作成
#      07/29    createTempCase:OF-v用を追加
#      08/07    remesh:createBaffleを削除
#

import os, sys
import shutil
import glob
import pyTreeFoam

#
#  createTempCase
#------------------
def createTempCase():
    """ tempCaseを作成する"""
    global caseDir, tempCaseDir, timeFolders, iniTime, configDict
    #tempCaseの「0」をクリアする
    fields = glob.glob(tempCaseDir + "/0/*")
    for field in fields:
        if os.path.isfile(field):
            os.remove(field)
        elif os.path.isdir(field):
            shutil.rmtree(field)
    # #fieldをtempCaseにコピー
    case = pyTreeFoam.case(caseDir)
    timeFolders = case.getTimeFolders()
    # if iniTime == "firstTime":
    #     gTime = timeFolders[0]
    # else:
    #     gTime = iniTime
    # sourceDir = caseDir + "/" + gTime
    # pasteDir = tempCaseDir + "/0"
    # copyFilesInFolder(sourceDir, pasteDir)
    #internalFieldをクリア
    tempCase = pyTreeFoam.case(tempCaseDir)
    fields = glob.glob(tempCaseDir + "/0/*")
    fieldDir = tempCaseDir + "/0"
    fields = tempCase.getFieldNames("0")
    tempCase.clearInternalFields(fieldDir, fields)
    #patchからstlを取得
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer > 9.0:
        #patchを取得
        if mapTime == "latestTime":
            maxTime = timeFolders[-1]
        else:
            maxTime = mapTime
        region = "."
        meshDir = case.getCurrMeshDir(maxTime, region, "boundary")
        patchConts = case.getPatchNameFromBoundary(meshDir)
        patches = list(map(lambda x: x[0], patchConts))
        #stlを取得
        run = pyTreeFoam.run(caseDir)
        for patch in patches:
            fileName = patch + ".stl"
            comm = "surfaceMeshTriangulate -patches '(" + patch + ")' "
            comm += fileName
            run.command(comm)
    elif OFversion[0] == "v":
        comm = "foamToSurface -latestTime -tri allPatches.stl"
        run = pyTreeFoam.run(caseDir)
        run.command(comm)
        patches = getEachPatchFiles("allPatches.stl")
    #stlをcopy
    for patch in patches:
        fileName = patch + ".stl"
        shutil.copy(fileName, tempCaseDir + "/constant/triSurface")
    #mapFieldsDictの有無チェック
    mapFieldsDict = tempCaseDir + "/system/mapFieldsDict"
    if os.path.exists(mapFieldsDict) == False:
        getFoamContents = pyTreeFoam.getFoamContents(tempCaseDir)
        sourceDir = getFoamContents.getOFversionFolder(OFversion)
        sourceFile = sourceDir + "/system/mapFieldsDict"
        pasteDir = tempCaseDir + "/system"
        shutil.copy(sourceFile, pasteDir)

#  getEachPatchFiles
def getEachPatchFiles(allPatchesFile):
    f = open(allPatchesFile); lines = f.readlines(); f.close()
    patches = []
    patch = ""
    patchLines = []
    for line in lines:
        if line[:len("solid ")] == "solid ":
            words = line.split()
            patch = words[1]
            patchLines = [line]
            patches.append(patch)
        elif line[:len("endsolid ")] == "endsolid ":
            patchLines.append(line)
            fileName = patch + ".stl"
            f = open(fileName, "w"); f.writelines(patchLines); f.close()
        else:
            patchLines.append(line)
    return patches

#
#  remesh
#----------
def remesh():
    """ メッシュを作成する"""
    global tempCaseDir, configDict
    run = pyTreeFoam.run(tempCaseDir)
    #mesh作成(versionで異なる)
    OFversion = configDict["OFversion"]
    numVer = pyTreeFoam.getNumVersion(OFversion)
    if numVer >= 8.0:
        comm = "surfaceFeatures"
    else:
        comm = "surfaceFeatureExtract"
    run.command(comm)
    #blockMesh
    comm = "blockMesh"
    run.command(comm)
    #snappyHexmesh
    comm = "snappyHexMesh -overwrite"
    run.command(comm)

#
#  mappingField
#----------
def mappingField():
    """ latestTimeの内容をnewMeshにmappingし、結果をcaseDirに戻す"""
    global caseDir, tempCaseDir, timeFolders, region, iniTime, mapTime
    #fieldをコピー    
    if iniTime == "firstTime":
        gTime = timeFolders[0]
    else:
        gTime = iniTime
    sourceDir = caseDir + "/" + gTime
    pasteDir = tempCaseDir + "/0"
    copyFilesInFolder(sourceDir, pasteDir)
    #mapping
    region = "."
    selRegion = "."
    if mapTime == "latestTime":    
        selTime = timeFolders[-1]
    else:
        selTime = mapTime
    sourceDir = caseDir
    comm  = "mapFields "
    comm += "-sourceRegion " + selRegion + " "
    comm += "-sourceTime " + selTime + " "
    comm += "-targetRegion " + selRegion + " "
    comm += sourceDir
    run = pyTreeFoam.run(tempCaseDir)
    run.command(comm)
    #mappingFieldをコピー
    #  caseDirのselTimeを保存
    print("backuping mapped time folder...")
    source = caseDir + "/" + selTime + "/" + region
    newDir = caseDir + "/" + selTime + ".org/" + region
    source = os.path.normpath(source)
    newDir = os.path.normpath(newDir)
    if os.path.exists(newDir) == True:
        shutil.rmtree(newDir)
    os.makedirs(newDir)
    copyFilesInFolder(source, newDir)
    #  caseDirのselTime内をクリア
    delFiles = glob.glob(source + "/*")
    for delFile in delFiles:
        if os.path.isdir(delFile) == True:
            shutil.rmtree(delFile)
        elif os.path.isfile(delFile) == True:
            os.remove(delFile)
    #  mapping結果をコピー
    print("copying mapped fields to time folder...")
    sourceDir = tempCaseDir + "/0/" + region
    pasteDir = caseDir + "/" + selTime + "/" + region
    copyFilesInFolder(sourceDir, pasteDir)
    #  selTime内のuniformフォルダをコピー
    sourceName = caseDir + "/" + selTime + ".org" + "/" + "uniform"
    pasteName = caseDir + "/" + selTime + "/" + "uniform"
    if os.path.exists(sourceName) == True:
        shutil.copytree(sourceName, pasteName)
    #  polyMeshをselTimeにコピー
    print("copying remeshed polyMesh to " + selTime + " foler...")
    sourceName = tempCaseDir + "/constant/polyMesh"
    pasteName = caseDir + "/" + selTime + "/" + region + "/polyMesh"
    if os.path.exists(pasteName) == True:
        shutil.rmtree(pasteName)
    shutil.copytree(sourceName, pasteName)


#  copyFilesInFolder
def copyFilesInFolder(sourceDir, pasteDir):
    """ folder内のfile、folderを全てcopy&pasteする
    copy側のfile名に「.unmapped」が付加されていれば削除してpasteする"""
    sourceFiles = glob.glob(sourceDir + "/*")
    for sourcePath in sourceFiles:
        name = os.path.basename(sourcePath)
        if os.path.isdir(sourcePath) == True:
            shutil.copytree(sourcePath, pasteDir + "/" + name)
        elif os.path.isfile(sourcePath) == True:
            pasteName = name
            if name[-len(".unmapped"):] == ".unmapped":
                pasteName = name[:-len(".unmapped")]
            shutil.copy(sourcePath, pasteDir + "/" + pasteName)

#
#  printHelp
def printHelp():
    cont = """
----------- remeshAndMapping.py -------------------------
latestTimeにて、patch形状をstl形式で読み取り、snappyHexMeshで
メッシュを切り直す。remesh後、field内容を新meshにmappingする。
remeshする為には、予めsnappyHexMeshでメッシュを作成するtempcase
を準備しておく。

remeshAndMapping.py [option]

<option>
-c, --case      :caseDirを指定（省略時は、currDir）
-t, --tempCase  :tempCaseDirを指定
                 このcaseでmesh作成、mappingを行う。
-i, --iniTime   :mappingするfieldを取得するtimeFolder
                 計算開始時のtimeFolderを指定する。
                 （省略時は、firstTime）
-m, --mapTime   :mappingの元dataを取得するtimeFolder
                 （省略時は、latestTime）
-h, --help      :helpを表示

<使用例>
remeshAndMapping.py
    :引数省略時は、「../<currDirのbasename>_mesh」をtempCaseとして設定する。
remeshAndMapping.py ../test_mesh
    :「../test_mesh」をtempCaseとして設定。
remeshAndMapping.py -t ../test_mesh
    :「../test_mesh」をtempCaseとして設定。"""
    print(cont)

#
#  errorCheckParameters
def errorCheckParameters():
    global caseDir, tempCaseDir, iniTime, mapTime
    """ 入力したparametersのerrorチェック"""
    flag = "OK"
    if os.path.exists(caseDir) == False:
        mess = "error: tempCaseDir '" + caseDir + "' is not exists!!!"
        print()
        print(mess)
        flag = "NG"
    elif os.path.exists(tempCaseDir) == False:
        mess = "error: tempCaseDir '" + tempCaseDir + "' is not exists!!!"
        print()
        print(mess)
        flag = "NG"
    elif iniTime != "firstTime":
        try:
            a = float(iniTime)
        except:
            mess = "error: iniTime '" + iniTime + "' is illegal value!!!"
            print()
            print(mess)
            flag = "NG"
    elif mapTime != "latestTime":
        try:
            a = float(mapTime)
        except:
            mess = "error: mapTime '" + mapTime + "' is illegal value!!!"
            print()
            print(mess)
            flag = "NG"
    return flag

#  getParameter
def getParameters():
    currDir = os.getcwd()
    dirname = os.path.dirname(currDir)
    tempCaseDir = dirname + "/" + os.path.basename(currDir) + "_mesh"
    caseDir = os.getcwd()
    mapTime = "latestTime"
    iniTime = "firstTime"
    i = 1
    while i < len(sys.argv):
        if sys.argv[i] == "-h" or sys.argv[i] == "--help":
            printHelp()
            exit()
        elif sys.argv[i] == "-c" or sys.argv[i] == "__case":
            i += 1
            caseDir = sys.argv[i]
        elif sys.argv[i] == "-t" or sys.argv[i] == "--tempCase":
            i += 1
            tempCaseDir = sys.argv[i]
        elif sys.argv[i] == "-i" or sys.argv[i] == "--iniTime":
            i += 1
            iniTime = sys.argv[i]
        elif sys.argv[i] == "-m" or sys.argv[i] == "--mapTime":
            i += 1
            mapTime = sys.argv[i]
        else:
            tempCaseDir = sys.argv[i]
        i += 1
    caseDir = os.path.abspath(caseDir)
    tempCaseDir = os.path.abspath(tempCaseDir)
    return caseDir, tempCaseDir, iniTime, mapTime

if __name__ == "__main__":
    import gettext
    localeDir = os.getenv("TreeFoamPath") + "/data/locale"
    gettext.install("treefoam", localeDir)

    #caseDir = os.getcwd()
    caseDir, tempCaseDir, iniTime, mapTime = getParameters()
    err = errorCheckParameters()
    if err != "OK":
        exit()
    configDict = pyTreeFoam.readConfigTreeFoam()
    createTempCase()
    remesh()
    mappingField()
    
